program TRegSvr;

{$APPTYPE CONSOLE}

uses
  SysUtils, Windows, ActiveX, ComObj, RegConst, ImageHlp;

type
  TRegType = (rtAxLib, rtTypeLib, rtExeLib, rtAssembly);
  TRegAction = (raReg, raUnreg);
  TRegProc = function : HResult; stdcall;
  TUnRegTlbProc = function (const libID: TGUID; wVerMajor, wVerMinor: Word;
    lcid: TLCID; syskind: TSysKind): HResult; stdcall;
  TAssemblyRegProc = function(FileName: PChar;
    Reserved: Pointer) : Boolean stdcall;

const
  ProcName: array[TRegAction] of PChar = (
    'DllRegisterServer', 'DllUnregisterServer');
  ExeFlags: array[TRegAction] of string = (' /regserver', ' /unregserver');
  AssemblyProcName: array[TRegAction] of PChar = (
    'RegisterAssembly', 'UnregisterAssembly');

var
  RegType: TRegType = rtAxLib;
  RegAction: TRegAction = raReg;
  QuietMode: Boolean = False;
  SideBySide: Boolean = False;
  FileName: string;
  RegProc: TRegProc;
  LibHandle: THandle;
  OleAutLib: THandle;
  UnRegTlbProc: TUnRegTlbProc;


procedure OutputStr(S: string);
begin
  if not QuietMode then 
  begin
    CharToOEM(PChar(S), PChar(S));
    Writeln(S);
  end;
end;

function IsAssembly(FileName: String): Boolean;
var
  Base: ^Byte;
  Handle, Map: HWND;
  DosHeader: PImageDosHeader;
  //NTHeaders: PImageNtHeaders;
  Size: LongWord;
  Extension: String;
begin
  Result := False;
  Extension := LowerCase(ExtractFileExt(FileName));
    
  if (FileExists(FileName) = True) and
   (CompareText(Extension, '.EXE') = 0) or
   (CompareText(Extension, '.DLL') = 0) then
  begin
    Handle := CreateFile(PChar(FileName), GENERIC_READ, FILE_SHARE_READ,
      nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    Map := CreateFileMapping(Handle, nil, PAGE_READONLY, 0, 0, nil);
    Base := MapViewOfFile(Map, FILE_MAP_READ, 0, 0, 0);
    DosHeader := PImageDosHeader(Base);

    if DosHeader = nil then
      Exit;

    if ImageDirectoryEntryToData(Base, False,
      IMAGE_DIRECTORY_ENTRY_COM_DESCRIPTOR, Size) = nil then
    begin
      if Handle <> 0 then
      begin
       UnmapViewOfFile(Base);
       CloseHandle(Map);
       CloseHandle(Handle);
      end;

      Exit;
    end;

    Result := True;

    if Handle <> 0 then
    begin
     UnmapViewOfFile(Base);
     CloseHandle(Map);
     CloseHandle(Handle);
    end;
  end;
end;

function DecodeOptions: Boolean;
var
  i: Integer;
  FileStart: Boolean;
  Param, FileExt: string;
begin
  Result := False;
  if ParamCount = 0 then Exit;
  FileName := '';
  for i := 1 to ParamCount do
  begin
    Param := ParamStr(i);
    FileStart := not (Param[1] in ['-', '/']);
    if FileStart then
    begin
      if FileName = '' then FileName := Param
      else FileName := FileName + ' ' + Param;
      // strip open and/or close quote if present
      if (FileName[1] = '"') then
      begin
        if (FileName[Length(FileName)] = '"') then
          FileName := Copy(FileName, 2, Length(FileName) - 2)
        else if FileName[1] = '"' then Delete(FileName, 1, 1);
      end;
    end
    else
    begin
      if Length(Param) < 2 then Exit;
      case Param[2] of
        'U', 'u': RegAction := raUnreg;
        'Q', 'q': QuietMode := True;
        'S', 's': SideBySide := True;
        'T', 't': RegType := rtTypeLib;
      end;
    end;
  end;
  FileExt := ExtractFileExt(FileName);
  if FileExt = '' then raise Exception.CreateFmt(SNeedFileExt, [FileName]);
  if RegType <> rtTypeLib then
  begin
    if CompareText(FileExt, '.TLB') = 0 then RegType := rtTypeLib
    else if CompareText(FileExt, '.EXE') = 0 then RegType := rtExeLib
    else if IsAssembly(FileName) then RegType := rtAssembly
    else RegType := rtAxLib;
  end;
  Result := True;
end;

procedure RegisterAxLib;
var
  CurrentDir,
  FilePath: string;
begin
  FilePath := ExtractFilePath(FileName);
  CurrentDir := GetCurrentDir;
  SetCurrentDir(FilePath);
  try
    LibHandle := LoadLibrary(PChar(FileName));
    if LibHandle = 0 then raise Exception.CreateFmt(SLoadFail, [FileName]);
    try
      @RegProc := GetProcAddress(LibHandle, ProcName[RegAction]);
      if @RegProc = nil then
        raise Exception.CreateFmt(SCantFindProc, [ProcName[RegAction],
          FileName]);
      if RegProc <> 0 then
        raise Exception.CreateFmt(SRegFail, [ProcName[RegAction], FileName]);
      OutputStr(Format(SRegSuccessful, [ProcName[RegAction]]))
    finally
      FreeLibrary(LibHandle);
    end;
  finally
    SetCurrentDir(CurrentDir);
  end;
end;

procedure RegisterTLB;
const
  RegMessage: array[TRegAction] of string = (SRegStr, SUnregStr);
var
  WFileName, DocName: WideString;
  TypeLib: ITypeLib;
  LibAttr: PTLibAttr;
  CurrentDir,
  TypeLibPath: string;
begin
  TypeLibPath := ExtractFilePath(FileName);
  CurrentDir := GetCurrentDir;
  if (TypeLibPath = '') and not SideBySide then
    FileName := IncludeTrailingPathDelimiter(CurrentDir) + FileName;
  if not FileExists(FileName) then
    raise Exception.CreateFmt(SFileNotFound, [FileName]);
  { Always make the directory with the tlb the current directory so that
    side by side registration and UNregistration will succeed. }
  if TypeLibPath <> '' then
    SetCurrentDir(TypeLibPath);
  if SideBySide then
    FileName := ExtractFileName(FileName);
  WFileName := FileName;
  OleCheck(LoadTypeLib(PWideChar(WFileName), TypeLib));
  OutputStr(Format(STlbName, [WFileName]));
  OleCheck(TypeLib.GetLibAttr(LibAttr));
  try
    OutputStr(Format(STlbGuid, [GuidToString(LibAttr^.Guid)]) + #13#10);
    if RegAction = raReg then
    begin
      OleCheck(TypeLib.GetDocumentation(-1, nil, nil, nil, @DocName));
      DocName := ExtractFilePath(DocName);
      OleCheck(RegisterTypeLib(TypeLib, PWideChar(WFileName), PWideChar(DocName)));
    end
    else begin
      OleAutLib := GetModuleHandle('OLEAUT32.DLL');
      if OleAutLib <> 0 then
        @UnRegTlbProc := GetProcAddress(OleAutLib, 'UnRegisterTypeLib');
      if @UnRegTlbProc = nil then raise Exception.Create(SCantUnregTlb);
      with LibAttr^ do
        OleCheck(UnRegTlbProc(Guid, wMajorVerNum, wMinorVerNum, LCID, SysKind));
    end;
  finally
    TypeLib.ReleaseTLibAttr(LibAttr);
    if TypeLibPath <> '' then
      SetCurrentDir(CurrentDir);
  end;
  OutputStr(Format(STlbRegSuccessful, [RegMessage[RegAction]]));
end;

procedure RegisterEXE;
var
  SI: TStartupInfo;
  PI: TProcessInformation;
  RegisterExitCode: BOOL;
begin
  FillChar(SI, SizeOf(SI), 0);
  SI.cb := SizeOf(SI);
  RegisterExitCode := Win32Check(CreateProcess(PChar(FileName), PChar(FileName + ExeFlags[RegAction]),
    nil, nil, True, 0, nil, nil, SI, PI));
  CloseHandle(PI.hThread);
  CloseHandle(PI.hProcess);
  if RegisterExitCode then
    OutputStr(Format(SExeRegSuccessful, [PChar(FileName + ExeFlags[RegAction])]))
  else
    OutputStr(Format(SExeRegUnsuccessful, [PChar(FileName + ExeFlags[RegAction])]));
end;

procedure RegisterAssembly;
const
  STRegAsm = 'tregasm.dll';
var
  RegProc: TAssemblyRegProc;
begin
  LibHandle := LoadLibrary(PChar(STRegAsm));

  if LibHandle = 0 then raise Exception.CreateFmt(SLoadFail, [STRegAsm]);
  try
    @RegProc := GetProcAddress(LibHandle, AssemblyProcName[RegAction]);
    if @RegProc = nil then
      raise Exception.CreateFmt(SCantFindProc, [AssemblyProcName[RegAction],
        STRegAsm]);
    if not RegProc(PChar(FileName), nil) then
      raise Exception.CreateFmt(SRegFail, [AssemblyProcName[RegAction], STRegAsm]);
    OutputStr(Format(SRegSuccessful, [AssemblyProcName[RegAction]]))
  finally
    FreeLibrary(LibHandle);
  end;
end;

begin
  try
    if not DecodeOptions then
      raise Exception.Create(SAbout + #13#10 + SUsage);
    OutputStr(SAbout);
    if not FileExists(FileName) then
      raise Exception.CreateFmt(SFileNotFound, [FileName]);
    case RegType of
      rtAxLib: RegisterAxLib;
      rtTypeLib: RegisterTLB;
      rtExeLib: RegisterEXE;
      rtAssembly: RegisterAssembly;
    end;
  except
    on E:Exception do OutputStr(E.Message);
  end;
end.
